﻿//Copyright (C) 2010  Jonathan Preece
//
//This program is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program.  If not, see <http://www.gnu.org/licenses/>.

using System;
using System.Collections.Generic;
using System.Drawing;

namespace RevisionAnalyser.Global.EasterEgg
{
    public class Snake : ISnake
    {
        private const int _width = 10;
        private const int _height = 10;

        #region Constructor

        public Snake()
        {
            _segments = new List<Segment>();
        }

        #endregion

        #region Implementation of ISnake

        /// <summary>
        /// Increases the size of the Snake by one segment
        /// </summary>
        public void Add()
        {
            var segment = new Segment(_width, _height);
            Segment tailSegment = _segments[0];

            switch (MovingDirection)
            {
                case Direction.Left:
                    segment.X = tailSegment.X - _width;
                    segment.Y = tailSegment.Y;
                    break;

                case Direction.Right:
                    segment.X = tailSegment.X + _width;
                    segment.Y = tailSegment.Y;
                    break;

                case Direction.Up:
                    segment.X = tailSegment.X;
                    segment.Y = tailSegment.Y + _height;
                    break;

                case Direction.Down:
                    segment.X = tailSegment.X;
                    segment.Y = tailSegment.Y - _height;
                    break;

                default:
                    throw new ArgumentOutOfRangeException();
            }

            _segments.Insert(0, segment);
        }

        /// <summary>
        /// Resets the Snake back to its starting size and position
        /// </summary>
        public void Reset()
        {
            _segments.Clear();
            MovingDirection = Direction.Left;

            int xPosition = 80;
            for (int i = 0; i < 5; i++)
            {
                var segment = new Segment(_width, _height, xPosition, 100);
                _segments.Add(segment);

                xPosition += 10;
            }
        }

        /// <summary>
        /// Moves the Snake in the given <see cref="ISnake.MovingDirection"/>.
        /// </summary>
        public void Move()
        {
            //Each segment needs to move independently
            //Start at the head, move that in the appropriate direction
            //Trickle the heads previous position down the chain

            //Take the heads starting position, move it down the chain
            for (int i = 0; i < HeadPosition - 1; i++)
            {
                _segments[i] = _segments[i + 1].Clone() as Segment;
            }

            switch (MovingDirection)
            {
                case Direction.Left:

                    //Reposition the head
                    _segments[HeadPosition - 1].X -= 10;

                    break;

                case Direction.Right:

                    _segments[HeadPosition - 1].X += 10;

                    break;

                case Direction.Up:

                    _segments[HeadPosition - 1].Y -= 10;

                    break;

                case Direction.Down:

                    _segments[HeadPosition - 1].Y += 10;

                    break;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        /// <summary>
        /// Returns all the segments of the Snake
        /// </summary>
        public List<Segment> ToSegments
        {
            get { return _segments; }
        }

        /// <summary>
        /// Gets the Snakes body
        /// </summary>
        public Rectangle[] GetSnakeBody
        {
            get
            {
                Rectangle[] body = new Rectangle[_segments.Count - 1];
                for (int i = 0; i < _segments.Count - 1; i++)
                    body[i] = _segments[i].ToRectangle();

                return body;
            }
        }

        /// <summary>
        /// Gets the Snakes body only, as segments
        /// </summary>
        public List<Segment> GetSnakeBodyAsSegments
        {
            get
            {
                return _segments.GetRange(0, _segments.Count - 1);
            }
        }

        /// <summary>
        /// Returns all segments as Rectangles so they can be drawn on the canvas
        /// </summary>
        public Rectangle[] ToRectangles
        {
            get
            {
                var rectangles = new Rectangle[_segments.Count];

                for (int i = 0; i < _segments.Count; i++)
                    rectangles[i] = _segments[i].ToRectangle();

                return rectangles;
            }
        }

        /// <summary>
        /// Gets or sets the direction in which the Snake is currently moving
        /// </summary>
        public Direction MovingDirection { get; set; }

        /// <summary>
        /// Gets the Snakes head segment
        /// </summary>
        public Segment GetHeadSegment()
        {
            return _segments[_segments.Count - 1];
        }

        /// <summary>
        /// Gets the width of each segment of the snake
        /// </summary>
        public int Width
        {
            get { return _width; }
        }

        /// <summary>
        /// Gets the height of each segment of the snake
        /// </summary>
        public int Height
        {
            get { return _height; }
        }

        #endregion

        #region Private Methods

        /// <summary>
        /// Gets the position of the head segment in the list
        /// </summary>
        private int HeadPosition
        {
            get { return _segments.Count; }
        }

        #endregion

        private readonly List<Segment> _segments;
    }
}